(*******************************************************************************
**
**  PREVCHEK
**
**  Component designed to be dropped onto the MAIN form of an Application
**  which will check to see if there is another copy of the Application also
**  running.  If there is the behaviour is controlled by the components
**  user set properties.
**
**  Originally written by Andy Strong - Compuserve ID   : 100716,3015
**                                      Internet Address: andrews@nbaqsl.co.uk
**
**  Released into the public domain 22/11/95
**
**
**  This component should be dropped on either the main form of the
**  application or on the Splash screen form, whichever shows first.
**
**  Controls:
**  This code may be used, modified, included in applications without any
**  license agreements as long as the disclaimers are accepted, and the
**  comments are left intact ( Some Hope! <g> ),
**
**  Disclaimer:
**  This software is released into the public domain on the strict understanding
**  that neither myself nor any associates or companies I work for have any
**  liability explictly or implied.
**
**  Possible Enhancements:
**  i)  Check that the form being dropped onto is marked as the MAINFORM of the
**      application.
**
*******************************************************************************)
unit Prevchek;

interface
uses
  WinTypes, WinProcs, Classes, SysUtils;

type
  PHWND = ^HWnd;                          { Pointer to Window Handle }
  { Define the special Method format for User intervention.
    If AbortInst is unchanged by the user method call, the current
    application instance will be terminated. }
  TSecondProc = procedure(Sender: TObject; var AbortInst: boolean) of object;

  { Define a duplicate object exception }
  TDuplicateComponent = class(Exception);
  { Define a Form not Owner object exception }
  TFormNotOwner = class(Exception);

  { The OneInstance Class declaration }
  TOneInstance = class(TComponent)
  private
    { Private declarations - Attributes }
    FActivatePrevious: boolean;         { Bring previous instance to front? }
    FDuplicateMessage: string;          { Custom Message if previous detected }
    FDisplayMessage: boolean;           { Display Message if previous detected? }
    FOnSecondInst: TSecondProc;         { User Method Pointer to be called }
    { Private declarations - Methods }
    procedure GotoPreviousInstance;     { Activates previous Instance of Application }
    procedure CheckForPrevious;         { Actually checks for previous instance }

  public
    { Public declarations - Methods }
    constructor Create(AOwner: TComponent); override;
    procedure Loaded; override;

  published
    { Published declarations }
    property ActivatePrevious: boolean read FActivatePrevious write FActivatePrevious;
    property DuplicateMessage: string read FDuplicateMessage write FDuplicateMessage;
    property DisplayMessage: boolean read FDisplayMessage write FDisplayMessage;
    property OnSecondInst: TSecondProc read FOnSecondInst write FOnSecondInst;
  end;

{ Unit EXPORTED procedure definition.  This procedure is declared OUTSIDE of the
  CLASS definition.  This is so that it can be declared EXPORT, which adds
  special precursor code compatible with the WINAPI call ENUMWINDOWS }
function EnumFunc(Wnd:HWND; TargetWindow:PHWND): bool; export;

procedure Register;                     { Register for Object Inspector }

implementation

uses
  Dialogs, Forms, Controls;

(*
**  Overridden constructor for our component.  Creates underlying TComponent,
**  sets our own method pointer to an unassigned state, and checks that there
**  is only one TOneInstance Component on the form.  If there is another one,
**  this one is destroyed by virtue of the exception handler within the design
**  mode.
*)
constructor TOneInstance.Create(AOwner: TComponent);
var
  i: word;				{ General Loop Counter }
  CompCount: byte;			{ TOneInstance Component count }
begin
  inherited Create(AOwner);             { Call TComponent Constructor }
  FOnSecondInst := nil;                 { Set our Method Pointer to UnAssigned }
  CompCount := 0;                       { Initialise Component Count to zero }
  { If we are designing at present }
  if (csDesigning in ComponentState) then
    if (AOwner is TForm) then
      with (AOwner as TForm) do
      begin
        for i := 0 to ComponentCount - 1 do { Check if there is already one of us! }
          if Components[i] is TOneInstance then
            inc(CompCount);

        if CompCount > 1 then           { One of them is us! }
          raise TDuplicateComponent.Create
                ('There is already a TOneInstance component on this Form');
      end
      else
        raise TFormNotOwner.Create
              ('The owner of TOneInstance Component is not a TForm');
end;

(*
**  Overridden LOADED method.  This ensures that the check for a previous
**  instance of our Application is done immediately after the TOneInstance
**  components properties have been read in from the form, and initialised.
*)
procedure TOneInstance.Loaded;
begin
  inherited Loaded;                     { Always call inherited Loaded method }
  CheckForPrevious;                     { May not return from here! }
end;

(*
**  The method which actually does the check for a previous running copy of
**  our application.  It handles the identification and actions according to
**  the user settings of its properties.
*)
procedure TOneInstance.CheckForPrevious;
var
  TerminateApp: boolean;                { Kills this instance if True }
  CurCursor: TCursor;                   { Save Current Cursor shape }
begin
  TerminateApp := True;                 { Assume we are going to Die }
  if HPrevInst <> 0 then                { Are we already running? }
    { Yes we are - but we don't care if we are in design mode! }
    if not (csDesigning in ComponentState) then
    begin
      CurCursor := Screen.Cursor;       { Save Current }
      Screen.Cursor := crDefault;       { Set to Default }
      if FDisplayMessage then           { Shall we display a Message? }
        if FDuplicateMessage <> '' then { Yes - Is there a custom message? }
          MessageDlg(FDuplicateMessage, mtInformation, [mbOK], 0)
        else                            { Display default message }
          MessageDlg('This Application is already running!', mtInformation, [mbOK], 0);

      { If the user has set a  procedure to be called when a second instance
        is identified, call it passing in the TerminateApp flag set to True }
      if Assigned(FOnSecondInst) then
        FOnSecondInst(Self, TerminateApp);
      { If the user wants to activate the previous instance of this application,
        this means implicitly that we MUST terminate this instance }
      if FActivatePrevious then
      begin
       GotoPreviousInstance;            { Bring up previous instance }
       TerminateApp := True;            { In case User says not! }
      end;

      if TerminateApp then              { Must we now kill ourselves? }
      begin
        Application.Terminate;          { Send Kill Message to ourselves }
        Application.ProcessMessages;    { Ensure the message is processed }
        Halt;                           { Stop NOW! }
      end;
      Screen.Cursor := CurCursor;       { Set back to original: we run! }
    end;
end;


(*
**  Procedure to activate an Applications previous Instance
**  The GOTOPREVIOUSINSTANCE & ENUMFUNC routines were kindly
**  supplied by Xavier Pacheco - TeamB (excluding comments <g>).
*)
procedure TOneInstance.GotoPreviousInstance;
var
  PrevInstWnd : HWND;
begin
  PrevInstWnd := 0;                             { Initialise Handle to zero }
  { ENumWindows calls ENUMFUNC for each of the Top Level windows that WINDOWS knows about }
  EnumWindows(@EnumFunc,longint(@PrevInstWnd)); { Scan Top Level windows - Calls ENumProc}
  if PrevInstWnd <> 0 then                      { We found our previous instance }
    if IsIconic(PrevInstWnd) then               { If ICONised restore it }
      ShowWindow(PrevInstWnd,SW_RESTORE)
    else
      BringWindowToTop(PrevInstWnd);            { Bring to top and Activate it }
end;

(*
**  This is a Windows 'CallBack' function.  The function 'GotoPreviousInstance'
**  passes it to ENUMWINDOWS.  ENumWindows scans all Top level windows, passing
**  back their HWND until either there are no more to parse, or this callback
**  function returns 0.
*)
function EnumFunc(Wnd:HWND; TargetWindow:PHWND): bool;
var
  ClassName : array[0..30] of char;
begin
  Result := true;                               { Ensure we scan the next one }
  { If the windows handle being scanned now is the same as our previous
    instance handle ... }
  if GetWindowWord(Wnd, GWW_HINSTANCE) = hPrevInst then
  begin
    GetClassName(Wnd,ClassName,30);             { Get the Class name of current }
    { All Delphi Apps have a 'TApplication' window hidden from view ... }
    if StrIComp(ClassName, 'TApplication') = 0 then 
    begin
      TargetWindow^ := Wnd;                     { This is the window to activate }
      Result := false;                          { Stop any more scans }
    end;
  end;
end;

(*
**  Registration of us for the Object Inspector & Component Palette
*)
procedure Register;
begin
  RegisterComponents('Custom', [TOneInstance]);
end;

end.
